﻿using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using CE.Domain.DataObjects.Sqlite;
using CE.Domain.Inventory.DataObjects.Tables;
using CE.Domain.ModelingObjects.Repositories;

namespace CE.Domain.Inventory.ModelingObjects.Repositories
{
	public class InventoryItemRepository : GenericTableRepository<InventoryItem>
	{
		#region Constructor
		public InventoryItemRepository(ISqliteDataConnection dataConnection, IDataRepository dataRepository)
			: base(InventoryTableNames.InventoryItem, dataConnection)
		{
			DataRepository = dataRepository;
		}
		#endregion

		#region Public Methods
		public override List<InventoryItem> LoadItems(Expression<Func<InventoryItem, bool>> predicate) => LoadItems(predicate, null);

		public List<InventoryItem> LoadItems(Expression<Func<InventoryItem, bool>> predicate, InventoryFilter filter)
		{
			var items = base.LoadItems(predicate);
			List<InventoryItem> retVal = new List<InventoryItem>();
			foreach (var item in items)
			{
				PostLoad(item);
				if (filter != null)
				{
					if (!filter.CheckAll(item))
						continue;
				}
				retVal.Add(item);
			}
			return retVal;
		}
		public override List<InventoryItem> LoadItems() => LoadItems(null);

		public List<InventoryItem> LoadItems(InventoryFilter filter)
		{
			var items = base.LoadItems();
			List<InventoryItem> retVal = new List<InventoryItem>();
			foreach (var item in items)
			{
				PostLoad(item);
				if (filter != null)
				{
					if (!filter.CheckAll(item))
						continue;
				}
				retVal.Add(item);
			}
			return retVal;
		}
		public override InventoryItem Load(int id)
		{
			var retVal = base.Load(id);
			PostLoad(retVal);
			return retVal;
		}
		#endregion

		#region Protected Methods
		protected override void PostLoad(InventoryItem item)
		{
			int roomId = item.RoomId;
			int locationId = item.LocationId;
			int boxId = item.BoxId;

			ITableRepository<Room> roomRepository = DataRepository.GetTableRepositoryFor<Room>(InventoryTableNames.Room);
			ITableRepository<Location> locationRepository = DataRepository.GetTableRepositoryFor<Location>(InventoryTableNames.Location);
			ITableRepository<Box> boxRepository = DataRepository.GetTableRepositoryFor<Box>(InventoryTableNames.Box);

			item.Room = new Room();
			if (roomRepository.Exists(r => r.Id == roomId))
				item.Room = roomRepository[roomId];

			item.Location = new Location();
			if (locationRepository.Exists(l => l.Id == locationId))
				item.Location = locationRepository[locationId];

			item.Box = new Box();
			if (boxRepository.Exists(b => b.Id == boxId))
				item.Box = boxRepository[boxId];
		}
		#endregion

		#region Private Properties
		private IDataRepository DataRepository { get; set; }
		#endregion
	}
}
